unit SAXWriter1;

{
  Demonstrate the generation of an XML document from a database
  using the SAX for Pascal writers.
  Requires 'movie-watcher' alias to be set up in BDE.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written December 3, 2002.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Db, DBTables, StdCtrls, ExtCtrls, CommonXML, SAX, SAXHelpers, SAXComps;

type
  TfrmWriterXML = class(TForm)
    memXML: TMemo;
    pnlButtons: TPanel;
      btnGenerate: TButton;
      btnSave: TButton;
    dlgSave: TSaveDialog;
    saxWriter: TSAXDocumentWriter;
    procedure btnGenerateClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
  private
  public
  end;

var
  frmWriterXML: TfrmWriterXML;

implementation

{$R *.DFM}

{ Generate the XML document as text }
procedure TfrmWriterXML.btnGenerateClick(Sender: TObject);
var
  Attributes: TAttributesImpl;
  Output: TStringStream;

  { Start a new element tag }
  procedure StartElement(Name: WideString);
  begin
    saxWriter.StartElement('', '', Name, Attributes);
    Attributes.clear;
  end;

  { End an element tag }
  procedure EndElement(Name: WideString);
  begin
    saxWriter.EndElement('', '', Name);
  end;

  { Save an attribute for adding to an element }
  procedure AddAttribute(Name, Value: WideString);
  begin
    Attributes.addAttribute('', '', Name, 'CDATA', Value);
  end;

  { Add a simple element that only contains text }
  procedure AddSimpleElement(Field: TField; AsCDATA: Boolean = False);
  begin
    StartElement(ModifyName(Field.FieldName));
    if AsCDATA then
      saxWriter.StartCData;
    saxWriter.Characters(Field.DisplayText);
    if AsCDATA then
      saxWriter.EndCDATA;
    EndElement(ModifyName(Field.FieldName));
  end;

  { Include empty field tag only if flag in DB set }
  procedure AddOptElement(Field: TField);
  begin
    if Field.AsBoolean then
    begin
      StartElement(ModifyName(Field.FieldName));
      EndElement(ModifyName(Field.FieldName));
    end;
  end;

  { Compile elements for the stars of the movie }
  procedure GenerateStars;
  begin
    with datCommonXML.qryStars do
    begin
      StartElement(StarringTag);
      First;
      while not EOF do
      begin
        AddSimpleElement(FieldByName(StarField));
        Next;
      end;
      EndElement(StarringTag);
    end;
  end;

  { Generate elements for each movie }
  procedure GenerateMovies;
  var
    BaseId: string;
  begin
    StartElement(MoviesTag);
    with datCommonXML.qryMovie do
    begin
      First;
      while not EOF do
      begin
        BaseId := FieldByName(MovieIdField).DisplayText;
        AddAttribute(Id, BaseId);
        AddAttribute(Rating, FieldByName(RatingField).DisplayText);
        if FieldByName(LogoURLField).AsString <> '' then
          AddAttribute(ModifyName(FieldByName(LogoURLField).FieldName),
            BaseId + 'Logo');
        if FieldByName(URLField).AsString <> '' then
          AddAttribute(ModifyName(FieldByName(URLField).FieldName),
            BaseId + 'Url');
        StartElement(MovieTag);
        AddSimpleElement(FieldByName(NameField));
        AddSimpleElement(FieldByName(LengthField));
        AddSimpleElement(FieldByName(DirectorField));
        GenerateStars;
        AddSimpleElement(FieldByName(SynopsisField), True);
        EndElement(MovieTag);
        Next;
      end;
    end;
    EndElement(MoviesTag);
  end;

  { Compile elements for the pricing schemes }
  procedure GeneratePricing;
  begin
    with datCommonXML.qryPricing do
    begin
      StartElement(PricingTag);
      First;
      while not EOF do
      begin
        AddAttribute(Id, FieldByName(PricingIdField).DisplayText);
        StartElement(PriceTag);
        AddSimpleElement(FieldByName(NameField));
        AddSimpleElement(FieldByName(PeriodField));
        AddSimpleElement(FieldByName(AdultField));
        AddSimpleElement(FieldByName(ChildField));
        AddSimpleElement(FieldByName(DiscountField));
        EndElement(PriceTag);
        Next;
      end;
      EndElement(PricingTag);
    end;
  end;

  { Generate elements for each cinema }
  procedure GenerateCinemas;
  begin
    StartElement(CinemasTag);
    with datCommonXML.qryCinema do
    begin
      First;
      while not EOF do
      begin
        AddAttribute(Id, FieldByName(CinemaIdField).DisplayText);
        StartElement(CinemaTag);
        AddSimpleElement(FieldByName(NameField));
        AddSimpleElement(FieldByName(PhoneField));
        AddSimpleElement(FieldByName(AddressField));
        AddSimpleElement(FieldByName(DirectionsField));
        StartElement(FacilitiesTag);
        AddOptElement(FieldByName(CandyBarField));
        AddOptElement(FieldByName(DisabledField));
        EndElement(FacilitiesTag);
        GeneratePricing;
        EndElement(CinemaTag);
        Next;
      end;
    end;
    EndElement(CinemasTag);
  end;

  { Compile elements for the sessions for each screening }
  procedure GenerateSessions;
  begin
    with datCommonXML.qrySessions do
    begin
      StartElement(SessionsTag);
      First;
      while not EOF do
      begin
        AddAttribute(PricingId, FieldByName(PricingIdField).DisplayText);
        StartElement(SessionTag);
        saxWriter.Characters(FieldByName(TimeField).DisplayText);
        EndElement(SessionTag);
        Next;
      end;
      EndElement(SessionsTag);
    end;
  end;

  { Generate elements for each screening }
  procedure GenerateScreenings;
  begin
    StartElement(ScreeningsTag);
    with datCommonXML.qryScreening do
    begin
      First;
      while not EOF do
      begin
        AddAttribute(MovieId, FieldByName(MovieIdField).DisplayText);
        AddAttribute(CinemaId, FieldByName(CinemaIdField).DisplayText);
        StartElement(ScreeningTag);
        AddSimpleElement(FieldByName(StartDateField));
        AddSimpleElement(FieldByName(EndDateField));
        StartElement(FeaturesTag);
        AddSimpleElement(FieldByName(DigSoundField));
        EndElement(FeaturesTag);
        StartElement(RestrictionsTag);
        AddOptElement(FieldByName(NoPassesField));
        EndElement(RestrictionsTag);
        GenerateSessions;
        EndElement(ScreeningTag);
        Next;
      end;
    end;
    EndElement(ScreeningsTag);
  end;

  { Generate DTD and contents }
  procedure GenerateDTD;
  var
    BaseId: string;
  begin
    saxWriter.StartDTD(MovieWatcherTag, '', XMLDTDFile);
    saxWriter.NotationDecl(JPEGType, JPEGPubId, JPEGSysId);
    saxWriter.NotationDecl(HTMLType, HTMLPubId, HTMLSysId);
    with datCommonXML.qryMovie do
    begin
      First;
      while not EOF do
      begin
        BaseId := FieldByName(MovieIdField).DisplayText;
        if FieldByName(LogoURLField).AsString <> '' then
          saxWriter.UnparsedEntityDecl(BaseId + 'Logo', '',
            FieldByName(LogoURLField).DisplayText, JPEGType);
        if FieldByName(URLField).AsString <> '' then
          saxWriter.UnparsedEntityDecl(BaseId + 'Url', '',
            FieldByName(URLField).DisplayText, HTMLType);
        Next;
      end;
    end;
    saxWriter.EndDTD;
  end;

  { Generate XML prolog, style sheet reference, and main element }
  procedure GenerateDocument;
  begin
    saxWriter.StartDocument;
    GenerateDTD;
    saxWriter.Comment(XMLComment);
    saxWriter.ProcessingInstruction(XMLStyleTag, XMLStyleAttrs);
    StartElement(MovieWatcherTag);
    GenerateMovies;
    GenerateCinemas;
    GenerateScreenings;
    EndElement(MovieWatcherTag);
    saxWriter.EndDocument;
  end;

begin
  Screen.Cursor       := crHourGlass;
  btnGenerate.Enabled := False;
  try
    memXML.Lines.Clear;
    Attributes := TAttributesImpl.Create;
    try
      Output := TStringStream.Create('');
      try
        saxWriter.Stream := Output;
        { Generate the structure }
        GenerateDocument;
        Output.Position := 0;
        memXML.Lines.LoadFromStream(Output);
      finally
        Output.Free;
      end;
    finally
      Attributes.Free;
    end;
  finally
    btnGenerate.Enabled := True;
    Screen.Cursor       := crDefault;
  end;
end;

{ Save the generated XML }
procedure TfrmWriterXML.btnSaveClick(Sender: TObject);
begin
  with dlgSave do
    if Execute then
      memXML.Lines.SaveToFile(Filename);
end;

end.
